library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

entity dp_ram_2048x16_tb is
end entity;

architecture sim of dp_ram_2048x16_tb is
    constant ADDR_WIDTH : natural := 11;
    constant DATA_WIDTH : natural := 16;
    constant DEPTH      : natural := 2**ADDR_WIDTH;

    signal wclk      : std_logic := '0';
    signal rclk      : std_logic := '0';
    signal write_en  : std_logic := '0';
    signal waddr     : std_logic_vector(ADDR_WIDTH-1 downto 0) := (others => '0');
    signal raddr     : std_logic_vector(ADDR_WIDTH-1 downto 0) := (others => '0');
    signal din       : std_logic_vector(DATA_WIDTH-1 downto 0) := (others => '0');
    signal dout      : std_logic_vector(DATA_WIDTH-1 downto 0);

begin
    -------------------------------------------------------------------------
    -- DUT
    -------------------------------------------------------------------------
    DUT: entity work.dp_ram_2048x16
        generic map (
            addr_width => ADDR_WIDTH,
            data_width => DATA_WIDTH
        )
        port map (
            write_en => write_en,
            waddr    => waddr,
            wclk     => wclk,
            raddr    => raddr,
            rclk     => rclk,
            din      => din,
            dout     => dout
        );

    -------------------------------------------------------------------------
    -- Clocks
    -------------------------------------------------------------------------
    wclk <= not wclk after 5 ns;
    rclk <= not rclk after 5 ns;

    -------------------------------------------------------------------------
    -- Read sine wave, then (after gap) read triangle
    -------------------------------------------------------------------------
    read_proc : process
    begin
        -- prepare address 0 and wait one rising edge so dout is valid for first sample
        raddr <= (others => '0');
        wait until rising_edge(rclk);

        -- read full sine (address changes on falling edge, read on rising edge)
        for i in 0 to DEPTH-1 loop
            wait until falling_edge(rclk);
            raddr <= conv_std_logic_vector(i+1, ADDR_WIDTH);
            wait until rising_edge(rclk);
            -- dout is available here for sample i
        end loop;

        -- small gap to separate phases
        wait for 50 ns;

        -- prepare to read triangle from address 0
        raddr <= (others => '0');
        wait until rising_edge(rclk);

        -- read full triangle
        for i in 0 to DEPTH-1 loop
            wait until falling_edge(rclk);
            raddr <= conv_std_logic_vector(i+1, ADDR_WIDTH);
            wait until rising_edge(rclk);
            -- dout is available here for triangle sample i
        end loop;

        wait;
    end process;

    -------------------------------------------------------------------------
    -- Write triangle wave with proper shape (writes while sine is being read)
    -------------------------------------------------------------------------
    write_proc : process
        variable val : integer;
    begin
        wait for 30 ns; -- small offset after sine reading starts
        for i in 0 to DEPTH-1 loop
            wait until falling_edge(wclk);   -- set addr/data on falling edge
            write_en <= '1';
            waddr <= conv_std_logic_vector(i, ADDR_WIDTH);

            if i < (DEPTH/2) then
                val := integer((i * 65535) / (DEPTH/2 - 1));
            else
                val := integer(((DEPTH - 1 - i) * 65535) / (DEPTH/2 - 1));
            end if;

            din <= conv_std_logic_vector(val, DATA_WIDTH);
            wait until rising_edge(wclk);   -- write on rising edge
        end loop;
        write_en <= '0';
        wait;
    end process;

end architecture;
